#include <core.p4>
#include <xsa.p4>

#define MAX_HOPS 6
#define TSTMP_THRESHOLD 3000
const bit<8> PROTO_SRV6 = 253;

header ethernet_t {
    bit<48> dstAddr;      // Destination MAC address.
    bit<48> srcAddr;      // Source MAC address.
    bit<16> ether_type;    // Ethernet type
}

header ipv6_t {
    bit<4> version; 
    bit<8> traffic_class;
    bit<20> flow_label;
    bit<16> payload_len;
    bit<8> next_hdr;
    bit<8> hop_limit;
    bit<128> src_addr;
    bit<128> dst_addr;
}

header srv6_ra_t {
    bit<8> next_hdr;
    bit<8> hdr_ext_len;
    bit<8> routing_type;
    bit<8> segment_left;
    bit<8> last_entry;
    bit<8> flags;
    bit<16> tag;
}

header ra_list_t{
   bit<128> ra_element;
}

struct headers {
    ethernet_t ethernet;
    ipv6_t     ipv6;
    srv6_ra_t  srv6_ra;
    ra_list_t[MAX_HOPS] ra_list;
}

struct extern_in {
    bit<8>  reset_event_id;
    bit<64> timestamp;
    bit<64> dst_dmz_key;
    bit<8>  event_id; 
}

struct smartnic_metadata {
    bit<64> timestamp_ns;    // 64b timestamp (nanoseconds). Set when the packet arrives.
    bit<16> pid;             // 16b packet id for platform (READ ONLY - DO NOT EDIT).
    bit<3>  ingress_port;    // 3b ingress port (0:CMAC0, 1:CMAC1, 2:HOST0, 3:HOST1).
    bit<3>  egress_port;     // 3b egress port (0:CMAC0, 1:CMAC1, 2:HOST0, 3:HOST1).
    bit<1>  truncate_enable; // Reserved (tied to 0).
    bit<16> truncate_length; // Reserved (tied to 0).
    bit<1>  rss_enable;      // Reserved (tied to 0).
    bit<12> rss_entropy;     // Reserved (tied to 0).
    bit<4>  drop_reason;     // Reserved (tied to 0).
    bit<32> scratch;         // Reserved (tied to 0).
}

// ****************************************************************************** //
// *************************** P A R S E R  ************************************* //
// ****************************************************************************** //

parser ParserImpl(packet_in packet,
                   out headers hdr,
                   inout smartnic_metadata sn_meta,
                   inout standard_metadata_t smeta) {
    state start {
        transition parse_ethernet;
    }

    state parse_ethernet {
        packet.extract(hdr.ethernet);
        transition select(hdr.ethernet.ether_type) {
            0x86dd: parse_ipv6;
        }
    }

    state parse_ipv6 {
        packet.extract(hdr.ipv6); transition select(hdr.ipv6.next_hdr) {
            PROTO_SRV6: parse_srv6_ra;
            default: accept;
        } 
    }

    state parse_srv6_ra {
        packet.extract(hdr.srv6_ra); 
        transition parse_ra_list;
    }

    state parse_ra_list {
        packet.extract(hdr.ra_list.next); 
        transition select(hdr.ra_list.last.ra_element) {
            0: accept; 
            default: parse_ra_list;
        }
    }
}


// ****************************************************************************** //
// **************************  P R O C E S S I N G   **************************** //
// ****************************************************************************** //

// Define the processing logic for matching and actions.
control MatchActionImpl(inout headers hdr,
                         inout smartnic_metadata sn_meta,
                         inout standard_metadata_t smeta) {
    bit<64> timestamp = 0;
    bit<32> hash_val = 0;
    bit<64> u280_dna = 0;

    bool is_ingress = false;
    bool is_egress = false;
    bool is_edge = false; 
    bool egress_entry_found = false;  
    bool is_to_wan = false;

    bit<40> max_event_cnt = 0;
    bit<64> destination_dmz_key = 0; 

    bit<64> decrypted_timestamp = 0;
    bit<8> event_id = 0;
    bit<8> reset_event_id = 0;
    bit<40> event_epoch = 0;
    bit<16> time_diff = 0;
    bit<128> ra_elem = 0; 
    bit<128> ra_elem_ing = 0; 
    bit<128> ra_elem_eg = 0; 
    bit<128> error_node_addr = 0;

    UserExtern<extern_in, bit<128>>(3) user_extern_encrypt;

    action dropPacket() {
        smeta.drop = 1;  // Set drop flag in standard metadata.
    }

    action get_u280_info(bit<64> dna, bit<1> ingress, bit<1> egress, bit<1> edg) {
        u280_dna = dna;   
        is_ingress = (ingress == 1); 
        is_egress = (egress == 1);  
        is_edge = (edg == 1); 
    }
    action format_ingress_ra(bit<64> ra_tstmp, bit<32> switch_hash) { 
        ra_elem_ing[127:64] = ra_tstmp;
        ra_elem_ing[63:32] = switch_hash;
        ra_elem_ing[31:16] = (bit<16>)(u280_dna & 0xffff);
    } 
    action format_egress_ra() {
        ra_elem_eg[15:0] = (bit<16>)(u280_dna & 0xffff);
    }
    action get_calendar(bit<8> id, bit<40> max_cnt) { 
        event_id = id;
        max_event_cnt = max_cnt;
    }
    action get_tstmp(bit<64> tstmp) { 
        timestamp = tstmp;
    }
    action get_dst_dmz_key(bit<64> dst_dmz_key, bit<1> to_wan) {  
        destination_dmz_key = dst_dmz_key; 
        is_to_wan = (to_wan == 1);
    }
    action get_error_node_addr(bit<128> err_addr) {
        error_node_addr = err_addr;
    }

    action push_stack() {
        hdr.ra_list[4] = hdr.ra_list[3];
        hdr.ra_list[3] = hdr.ra_list[2];
        hdr.ra_list[2] = hdr.ra_list[1];
        hdr.ra_list[1] = hdr.ra_list[0];
        hdr.ra_list[0].ra_element = 0xffffffffffffffffffffffffffffffff; 
    }

    action get_reset_event_id (bit<8> reset_id){
        reset_event_id = reset_id;
    }
    
    table u280_info {
        key = {
            hdr.ethernet.ether_type: exact;
        } 
        actions = {
            get_u280_info;
            NoAction;
        }
        size = 8;
        default_action = NoAction; 
    }

    table ra_ingress {
        key = {
            hdr.ethernet.ether_type: exact;
        }
        actions = {
            format_ingress_ra;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;
    }

    table ra_egress {
        key = {
            hdr.ra_list[0].ra_element[127:32]: exact;
        }
        actions = {
            format_egress_ra;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;
    }

    table clk_tbl { 
        key = {
            hdr.ethernet.ether_type: exact;
        } 
        actions = {
            get_tstmp;
            NoAction;
        }
        size = 8;
        default_action = NoAction; 
    }

    table dst_dmz_key {   
        key = {
            hdr.ipv6.src_addr: exact; 
            hdr.ipv6.dst_addr: exact; 
        } 
        actions = {
            get_dst_dmz_key;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;  
    }

    table calendar {   
        key = {
            hdr.ipv6.src_addr: exact; 
            hdr.ipv6.dst_addr: exact; 
        } 
        actions = {
            get_calendar;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;  
    }

    table calendar_reset {   
        key = {
            hdr.ethernet.ether_type: exact; //dummy key since key must exceed 10bit
            event_id: exact;
        } 
        actions = {
            get_reset_event_id;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;  
    }
    
    table error_node {
        key = {
            hdr.ethernet.ether_type: exact;
        } 
        actions = {
            get_error_node_addr;
            NoAction;
        }
        size = 1024;
        default_action = NoAction;  
    }
 
    extern_in ext_in;
    bit<128> temp;
    apply {
        if (smeta.parser_error != error.NoError) {
            dropPacket();
            return; 
        }
        u280_info.apply();
        clk_tbl.apply(); 
        ra_ingress.apply();
        ra_egress.apply();
        error_node.apply();
        bool dmz_key_hit = dst_dmz_key.apply().hit; 
        bool calendar_hit = calendar.apply().hit;
        bool calendar_reset_hit = calendar_reset.apply().hit;

        if (hdr.ipv6.isValid() && hdr.srv6_ra.isValid()) {
            if (is_ingress) {
                push_stack();
                hdr.ra_list[0].ra_element = ra_elem_ing;
            } else if (is_egress) {
                hdr.ra_list[0].ra_element = hdr.ra_list[0].ra_element + ra_elem_eg;
            } else if (!dmz_key_hit) {
                hdr.ipv6.dst_addr = error_node_addr; 
            } else if (is_edge && is_to_wan) {
                push_stack();
                ra_elem = 0;
                if (!calendar_hit) {
                     event_id = 0xff;
                     max_event_cnt = (bit<40>)(0xffffffffff); 
                     hdr.ipv6.dst_addr = error_node_addr;
                } 
                if (calendar_reset_hit) {
                    ext_in.reset_event_id = reset_event_id; 
                }
                ext_in.timestamp = timestamp; 
                ext_in.dst_dmz_key = destination_dmz_key;
                ext_in.event_id = event_id;
                user_extern_encrypt.apply(ext_in, ra_elem);
                hdr.ra_list[0].ra_element = ra_elem;
            } else if (is_edge) {
                if (calendar_reset_hit) {
                    ext_in.reset_event_id = reset_event_id; 
                }
                ext_in.timestamp = hdr.ra_list[0].ra_element[127:64];
                ext_in.dst_dmz_key = u280_dna;  //using current u280 dna as key
                ext_in.event_id = hdr.ra_list[0].ra_element[63:56];
                user_extern_encrypt.apply(ext_in, ra_elem);
                push_stack();
                hdr.ra_list[0].ra_element = ra_elem;
                if (timestamp - ra_elem[127:64] > TSTMP_THRESHOLD) { 
                    hdr.ipv6.dst_addr = error_node_addr;
                } 
                //hdr.ra_list[MAX_HOPS-1].ra_element[127:120]  = hdr.ra_list[0].ra_element[63:56];
                //hdr.ra_list[MAX_HOPS-1].ra_element[111:96]   = hdr.ra_list[0].ra_element[15:0]; 
                //hdr.ra_list[MAX_HOPS-1].ra_element[95:80]    = ra_elem[15:0];
                //hdr.ra_list[MAX_HOPS-1].ra_element[79:40]    = hdr.ra_list[0].ra_element[55:16];
                //hdr.ra_list[MAX_HOPS-1].ra_element[39:0]     = ra_elem[55:16]; 
            } 
        } else if (hdr.ipv6.isValid() && hdr.ipv6.next_hdr != 0x3a) {   
            hdr.srv6_ra.setValid();
            hdr.srv6_ra.next_hdr = hdr.ipv6.next_hdr;
            hdr.ipv6.next_hdr = PROTO_SRV6;
            hdr.srv6_ra.routing_type = 4; 
            hdr.srv6_ra.last_entry = (MAX_HOPS - 1);
            hdr.srv6_ra.flags = 0;

            hdr.srv6_ra.tag = 0;
            hdr.ra_list[0].setValid();
            hdr.ra_list[1].setValid();
            hdr.ra_list[2].setValid();
            hdr.ra_list[3].setValid();
            hdr.ra_list[4].setValid();
            hdr.ra_list[5].setValid();
            hdr.ra_list[0].ra_element = 0xffffffffffffffffffffffffffffffff; 
            hdr.ra_list[1].ra_element = 0xffffffffffffffffffffffffffffffff; 
            hdr.ra_list[2].ra_element = 0xffffffffffffffffffffffffffffffff; 
            hdr.ra_list[3].ra_element = 0xffffffffffffffffffffffffffffffff; 
            hdr.ra_list[4].ra_element = 0xffffffffffffffffffffffffffffffff; 
            hdr.ra_list[5].ra_element = 0x0;

            hdr.ipv6.payload_len = hdr.ipv6.payload_len + 104;
            hdr.srv6_ra.hdr_ext_len = hdr.srv6_ra.hdr_ext_len + 11;

            if (is_ingress) {
                push_stack();
                hdr.ra_list[0].ra_element = ra_elem_ing;
            } else if (is_egress) {
                hdr.ra_list[0].ra_element = hdr.ra_list[0].ra_element + ra_elem_eg;
            } else if(!dmz_key_hit) {
                hdr.ipv6.dst_addr = error_node_addr; 
            } else if (is_edge && is_to_wan) {
                push_stack();
                ra_elem = 0;
                if (!calendar_hit) {
                     event_id = 0xff;
                     max_event_cnt = (bit<40>)(0xffffffffff); 
                     hdr.ipv6.dst_addr = error_node_addr;
                } 
                if (calendar_reset_hit) {
                    ext_in.reset_event_id = reset_event_id; 
                }
                ext_in.timestamp = timestamp; 
                ext_in.dst_dmz_key = destination_dmz_key;
                ext_in.event_id = event_id;
                user_extern_encrypt.apply(ext_in, ra_elem);
                hdr.ra_list[0].ra_element = ra_elem;
            } else if (is_edge) {
                if (calendar_reset_hit) {
                    ext_in.reset_event_id = reset_event_id; 
                }
                ext_in.timestamp = hdr.ra_list[0].ra_element[127:64];
                ext_in.dst_dmz_key = u280_dna;
                ext_in.event_id = hdr.ra_list[0].ra_element[63:56];
                user_extern_encrypt.apply(ext_in, ra_elem);
                push_stack();
                hdr.ra_list[0].ra_element = ra_elem;
                if (timestamp - ra_elem[127:64] > TSTMP_THRESHOLD) { 
                    hdr.ipv6.dst_addr = error_node_addr;
                } 
                //hdr.ra_list[MAX_HOPS-1].ra_element[127:120]  = hdr.ra_list[0].ra_element[63:56];
                //hdr.ra_list[MAX_HOPS-1].ra_element[111:96]   = hdr.ra_list[0].ra_element[15:0];  
                //hdr.ra_list[MAX_HOPS-1].ra_element[95:80]    = ra_elem[15:0];
                //hdr.ra_list[MAX_HOPS-1].ra_element[79:40]    = hdr.ra_list[0].ra_element[55:16];
                //hdr.ra_list[MAX_HOPS-1].ra_element[39:0]     = ra_elem[55:16]; 
            }
        }
    }
}

// ****************************************************************************** //
// ***************************  D E P A R S E R  ******************************** //
// ****************************************************************************** //

control DeparserImpl(packet_out packet,
                      in headers hdr,
                      inout smartnic_metadata sn_meta,
                      inout standard_metadata_t smeta) {
    apply {
        packet.emit(hdr.ethernet);
        packet.emit(hdr.ipv6);
        packet.emit(hdr.srv6_ra);
        packet.emit(hdr.ra_list);
    }
}

// ****************************************************************************** //
// *******************************  M A I N  ************************************ //
// ****************************************************************************** //

// Define the main pipeline.
XilinxPipeline(
    ParserImpl(), 
    MatchActionImpl(), 
    DeparserImpl()
) main;
