/* -*- P4_16 -*- */
#include <core.p4>
#include <v1model.p4>

const bit<16> TYPE_IPV4 = 0x800;
const bit<8> PROT_TCP = 6;
const bit<8> PROT_UDP = 17;
#define GTPU_PORT 2152

typedef bit<9>  egressSpec_t;
typedef bit<48> macAddr_t;
typedef bit<32> ip4Addr_t;
typedef bit<16> protport_t;
typedef bit<32> TEID;


/*************************************************************************
*********************** H E A D E R S  ***********************************
*************************************************************************/

header ethernet_t {
    macAddr_t dstAddr;
    macAddr_t srcAddr;
    bit<16>   etherType;
}

header ipv4_t {
    bit<4>    version;
    bit<4>    ihl;
    bit<8>    diffserv;
    bit<16>   length;
    bit<16>   id;
    bit<3>    flags;
    bit<13>   fragOffset;
    bit<8>    ttl;
    bit<8>    protocol;
    bit<16>   hdrChecksum;
    ip4Addr_t srcAddr;
    ip4Addr_t dstAddr;
}

header tcp_t {
    protport_t sport;
    protport_t dport;
    bit<32> seq;
    bit<32> ack;
    bit<4> offset;
    bit<4> reserved;
    bit<8> flags;
    bit<16> window;
    bit<16> chksum;
    bit<16> urg;
}

header udp_t {
    protport_t sport;
    protport_t dport;
    bit<16> length;
    bit<16> chksum;
}

header gtpu_h {
    bit<3> version;
    bit<1> protocol;
    bit<1> reserved;
    bit<1> extension;
    bit<1> sequence;
    bit<1> npdu;
    bit<8> type;
    bit<16> length;
    TEID teid;
}

struct metadata {
    /* empty */
}

struct headers {
    ethernet_t   ethernet;
    ipv4_t       ipv4;
    tcp_t        tcp;
    udp_t        udp;
    gtpu_h       gtpu;
    ipv4_t       ipv4Inner;
    udp_t        udpInner;
}

/*************************************************************************
*********************** P A R S E R  ***********************************
*************************************************************************/

parser MyParser(packet_in packet,
                out headers hdr,
                inout metadata meta,
                inout standard_metadata_t standard_metadata) {

    state start {
        transition parse_ethernet;
    }

    state parse_ethernet {
        packet.extract(hdr.ethernet);
        transition select(hdr.ethernet.etherType) {
            TYPE_IPV4: parse_ipv4;
            default: accept;
        }
    }

    state parse_ipv4 {
        packet.extract(hdr.ipv4);
        transition select(hdr.ipv4.protocol) {
            PROT_UDP: parse_udp;
            default: accept;
        }
    }

    state parse_udp {
        packet.extract(hdr.udp);
        transition select(hdr.udp.sport) {
            GTPU_PORT: parse_gtpu;
            default: accept;
        }
    }

    state parse_gtpu {
        packet.extract(hdr.gtpu);
        transition parse_ipv4Inner;
    }

    state parse_ipv4Inner {
        packet.extract(hdr.ipv4Inner);
        transition select(hdr.ipv4Inner.protocol) {
            PROT_UDP: parse_udp;
            default: accept;
        }
    }

    state parse_udpInner {
        packet.extract(hdr.udpInner);
        transition accept;
    }
}

/*************************************************************************
************   C H E C K S U M    V E R I F I C A T I O N   *************
*************************************************************************/

control MyVerifyChecksum(inout headers hdr, inout metadata meta) {   
    apply {

    }
}

/*************************************************************************
**************  I N G R E S S   P R O C E S S I N G   *******************
*************************************************************************/

control MyIngress(inout headers hdr,
                  inout metadata meta,
                  inout standard_metadata_t standard_metadata) {
   
    action forward_remove_tunnel(macAddr_t dstAddr, macAddr_t srcAddr, egressSpec_t port) {
        //Strip tunnel headers
        hdr.ipv4.setInvalid();
        hdr.udp.setInvalid();
        hdr.gtpu.setInvalid();
        //Push
        hdr.ethernet.srcAddr = srcAddr;
        hdr.ethernet.dstAddr = dstAddr;
        hdr.ipv4Inner.ttl = hdr.ipv4Inner.ttl - 1;
        standard_metadata.egress_spec = port;
    }

    action drop() {
        mark_to_drop(standard_metadata);
    }

    action forward_add_tunnel(macAddr_t dstAddr, macAddr_t srcAddr, egressSpec_t port,
                       ip4Addr_t srcTunnel, ip4Addr_t dstTunnel, TEID tunnelteid) {
        //Add tunnel header
        hdr.gtpu.setValid();
        hdr.gtpu.version = 1;
        hdr.gtpu.protocol = 1;
        hdr.gtpu.reserved = 0;
        hdr.gtpu.extension = 0;
        hdr.gtpu.sequence = 0;
        hdr.gtpu.npdu = 0;
        hdr.gtpu.type = 0xFF;
        hdr.gtpu.length = hdr.ipv4.length;
        hdr.gtpu.teid = tunnelteid;
        //Move current IP headers to inside
        hdr.ipv4Inner.setValid();
        hdr.ipv4Inner = hdr.ipv4;
        hdr.udpInner.setValid();
        hdr.udpInner = hdr.udp;
        //Set new IP headers
        hdr.ipv4.srcAddr = srcTunnel;
        hdr.ipv4.dstAddr = dstTunnel;
        hdr.ipv4.id = hdr.ipv4Inner.id;
        hdr.ipv4.protocol = PROT_UDP;
        hdr.udp.setValid();
        hdr.udp.sport = GTPU_PORT;
        hdr.udp.dport = GTPU_PORT;
        hdr.udp.length = hdr.gtpu.length + 16;
        hdr.udp.chksum = 0;
        hdr.ipv4.length = hdr.udp.length + 20;
        //Push
        hdr.ethernet.srcAddr = srcAddr;
        hdr.ethernet.dstAddr = dstAddr;
        standard_metadata.egress_spec = port;
    }
    
    action forward(macAddr_t dstAddr, macAddr_t srcAddr, egressSpec_t port) {
        hdr.ethernet.dstAddr = dstAddr;
        hdr.ethernet.srcAddr = srcAddr;
        standard_metadata.egress_spec = port;
        hdr.ipv4.ttl = hdr.ipv4.ttl - 1;
    }

    bit<32> ctr;
    register<bit<32>>(1) r_ctr;

    table table_internet {
        key = {
            hdr.ipv4.dstAddr:lpm;
            standard_metadata.ingress_port:exact;
            ctr:exact;
        }
        actions = {
            forward;
            forward_add_tunnel;
            forward_remove_tunnel;
            drop;
        }
        size = 1024;
        default_action = drop();
    }

    table table_host {
        key = {
            hdr.ipv4.dstAddr:exact;
            standard_metadata.ingress_port:exact;
            ctr:exact;
        }
        actions = {
            forward;
            forward_add_tunnel;
            forward_remove_tunnel;
            drop;
        }
        size = 1024;
        default_action = drop();
    }

    apply {
        if(hdr.ipv4.isValid()) {
            r_ctr.read(ctr, 0);
            if(table_host.apply().miss) {
                if(table_internet.apply().miss) {
                    drop();
                }
            }
            ctr = ctr + 1;
            if(ctr == 3) {
                ctr = 0;
            }
            r_ctr.write(0, ctr);
            return;
        }
        drop();
    }
}

/*************************************************************************
****************  E G R E S S   P R O C E S S I N G   *******************
*************************************************************************/

control MyEgress(inout headers hdr,
                 inout metadata meta,
                 inout standard_metadata_t standard_metadata) {
    apply {
    }
}

/*************************************************************************
*************   C H E C K S U M    C O M P U T A T I O N   **************
*************************************************************************/

control MyComputeChecksum(inout headers  hdr, inout metadata meta) {
    apply {  
        update_checksum(
            hdr.ipv4.isValid(),
            {
                hdr.ipv4.version,
                hdr.ipv4.ihl,
                hdr.ipv4.diffserv,
                hdr.ipv4.length,
                hdr.ipv4.id,
                hdr.ipv4.flags,
                hdr.ipv4.fragOffset,
                hdr.ipv4.ttl,
                hdr.ipv4.protocol,
                hdr.ipv4.srcAddr,
                hdr.ipv4.dstAddr
            },
            hdr.ipv4.hdrChecksum,
            HashAlgorithm.csum16
        );
        update_checksum(
            hdr.ipv4Inner.isValid(),
            {
                hdr.ipv4Inner.version,
                hdr.ipv4Inner.ihl,
                hdr.ipv4Inner.diffserv,
                hdr.ipv4Inner.length,
                hdr.ipv4Inner.id,
                hdr.ipv4Inner.flags,
                hdr.ipv4Inner.fragOffset,
                hdr.ipv4Inner.ttl,
                hdr.ipv4Inner.protocol,
                hdr.ipv4Inner.srcAddr,
                hdr.ipv4Inner.dstAddr
            },
            hdr.ipv4Inner.hdrChecksum,
            HashAlgorithm.csum16
        );
    }
}

/*************************************************************************
***********************  D E P A R S E R  *******************************
*************************************************************************/

control MyDeparser(packet_out packet, in headers hdr) {
    apply {
        packet.emit(hdr.ethernet);
        packet.emit(hdr.ipv4);
        packet.emit(hdr.udp);
        packet.emit(hdr.gtpu);
        packet.emit(hdr.ipv4Inner);
        packet.emit(hdr.udpInner);
    }
}

V1Switch(
MyParser(),
MyVerifyChecksum(),
MyIngress(),
MyEgress(),
MyComputeChecksum(),
MyDeparser()
) main;
