#!/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

import glob
import os
from pathlib import Path
from shutil import rmtree
from typing import Generator, List, cast

import click
from click import confirm

from dependencies import source
from utils.collections import flatten
from utils.filesystem_utils import list_all_files, delete_files_and_clean_dir_structure
from utils.install_dir_info import install_dir_info_manager, InstallDirCategory
from utils.log import p4studio_logs_dir
from utils.terminal import print_normal, print_separator, print_green
from workspace import current_workspace


@click.command("clean", short_help="Remove files generated by p4studio/cmake")
@click.option("--skip-logs", default=False, is_flag=True, help="Do not remove p4studio logs")
@click.option("--skip-build-dir", default=False, is_flag=True, help="Do not remove build files")
@click.option("--yes", '-y', default=False, is_flag=True, help="Do not prompt for confirmation")
@click.option("--install-dir", default=False, is_flag=True, help="Remove all installed components in install directory")
@click.option("--dependencies", default=False, is_flag=True, help="Remove dependencies installed in install directory")
def clean_command(skip_logs: bool, skip_build_dir: bool, yes: bool, install_dir: bool, dependencies: bool) -> None:
    """
    Remove files generated by p4studio/cmake as long as they were created in default locations.
    """
    print_normal("This will remove:")
    print_normal(" - temporary files created during dependency installation")
    if not skip_build_dir:
        print_normal(" - cmake build files")
    if not skip_logs:
        print_normal(" - p4studio logs")
    if install_dir:
        print_normal(" - installed components")
    if dependencies:
        print_normal(" - source dependencies")

    print_separator()

    if not yes:
        yes = confirm("Are you sure you want to continue?", default=False)

    if not yes:
        return

    print_green("Removing files...")
    if not skip_logs:
        remove_logs()
    if not skip_build_dir:
        remove_build_dir()
    if install_dir:
        remove_installed_components([InstallDirCategory.DEPENDENCIES])
    if dependencies:
        remove_source_dependencies()
    remove_temporary_files_from_dependencies_python_package()
    print_green("Files removed.")


def remove_source_dependencies() -> None:
    install_dir_info = install_dir_info_manager()
    installed_files = install_dir_info.files_of_category(InstallDirCategory.DEPENDENCIES)
    install_dir_info.clear_category(InstallDirCategory.DEPENDENCIES)
    delete_files_and_clean_dir_structure(
        [Path(x) for x in installed_files])


def remove_installed_components(manifest_categories_to_omit: List[InstallDirCategory]) -> None:
    workspace = current_workspace()
    install_dir_info = install_dir_info_manager()
    files_to_omit = flatten(
        cast(List[List[str]], [install_dir_info.files_of_category(c) for c in manifest_categories_to_omit]))
    files_in_install_dir = list_all_files(workspace.default_install_dir, recursively=True)
    files_to_delete = [f for f in files_in_install_dir if
                       str(f.absolute()) not in files_to_omit]
    delete_files_and_clean_dir_structure(files_to_delete)


def remove_logs() -> None:
    for file in glob.glob(str(p4studio_logs_dir() / '*.log')):
        Path(file).unlink()


def remove_build_dir() -> None:
    path = current_workspace().build_path
    if path.exists():
        rmtree(path.as_posix())


def remove_temporary_files_from_dependencies_python_package() -> None:
    for path in files_to_remove():
        if path.is_dir():
            rmtree(str(path))
        else:
            path.unlink()


def files_to_remove() -> Generator[Path, None, None]:
    package_dir = os.path.dirname(os.path.abspath(source.__file__))

    for pattern in ['PI*', 'boost*', 'bridge*', 'grpc*', 'thrift*', 'libcli*', 'Libcli*']:
        for file in glob.glob(package_dir + '/' + pattern):
            yield Path(file)
