#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

from pathlib import Path

from dependencies.source.source_dependency_config import SourceDependencyConfig
from utils.processes import execute

_BOOST_FILE = 'boost.tar.bz2'


def download_boost(config: SourceDependencyConfig) -> None:
    boost_attrs = config.dependency_manager().source_dependency_attributes("boost")
    download_dir = config.download_dir(ensure_exists=True)
    boost_file = download_dir / _BOOST_FILE

    if not boost_file.exists():
        execute("wget {} -nv -O {}".format(boost_attrs["url"], boost_file.as_posix()))


def install_boost(config: SourceDependencyConfig) -> None:
    boost_attrs = config.dependency_manager().source_dependency_attributes("boost")

    if not config.force and _is_boost_installed(config.install_dir, boost_attrs['version']):
        return

    download_boost(config)
    boost_file = (config.download_dir(ensure_exists=False) / _BOOST_FILE)
    build_dir = config.build_dir()

    execute("tar xf {} --strip-components 1 -C {}".format(boost_file, build_dir))
    execute("./bootstrap.sh --prefix={} --without-libraries=python".format(config.install_dir), build_dir)
    execute("./b2 -j{} {}".format(config.jobs, boost_attrs["flags"]), build_dir)
    execute("sudo ldconfig", build_dir)


def _is_boost_installed(path: Path, version: str) -> bool:
    required_boost_files = [
        'include/boost/variant.hpp',
        ('lib/libboost_system.so.%s' % version),
        ('lib/libboost_filesystem.so.%s' % version),
        ('lib/libboost_thread.so.%s' % version)
    ]

    return all((path / f).exists() for f in required_boost_files)
