#!/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.
import sys
from logging import FileHandler
from typing import Dict

import click
from click import ClickException, Abort

from app import app_command
from build import build_command
from clean.clean_command import clean_command
from config import configure_command
from dependencies import dependencies_command
from interactive.interactive_command import interactive_command
from packages.packages_command import packages_command
from profile import profile_command
from system.check_system_command import check_system_command
from utils.click import OrderedGroup
from utils.exceptions import ApplicationException
from utils.log import initialize_loggers, default_logger
from utils.p4studio_path import p4studio_path
from utils.yaml_utils import read_yaml_from_file
from workspace import configure_env_variables, in_workspace


@click.group(cls=OrderedGroup)
def p4studio_cli() -> None:
    """
    \b
    p4studio helps to manage SDE and its environment by:
    \b
    - installing dependencies,
    - building and installing SDE components,
    - building and installing P4 programs.

    \b
    If you do not know where to start, run:
      p4studio interactive
    """
    initialize_loggers()


P4STUDIO_COMMANDS = [
    interactive_command,
    profile_command,
    dependencies_command,
    configure_command,
    build_command,
    clean_command,
    check_system_command,
    app_command,
    packages_command,
]

for command in P4STUDIO_COMMANDS:
    p4studio_cli.add_command(command)


def default_options() -> Dict[str, object]:
    try:
        return read_yaml_from_file(p4studio_path().joinpath("defaults.yaml").as_posix())
    except FileNotFoundError:
        return {}


def p4studio_main() -> None:
    if in_workspace():
        configure_env_variables()
    try:
        p4studio_cli.main(
            sys.argv[1:],
            standalone_mode=False,
            default_map=default_options(),
            auto_envvar_prefix='P4STUDIO'
        )
    except (
            ClickException,
            ApplicationException,
            PermissionError
    ) as e:
        click.secho('Error: {}'.format(e), err=True, fg='red')

        for handler in default_logger().handlers:
            if isinstance(handler, FileHandler):
                click.secho("\nFor more details, see: {}".format(handler.baseFilename))

        sys.exit(1)
    except Abort:
        sys.exit(1)
