#!/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.
import os
from collections import OrderedDict
from typing import Optional, List, Sequence

import click
import yaml
from click import Choice
from click import Context
from click.utils import LazyFile
from yaml.representer import SafeRepresenter

from build import build_command
from config.config_option_utils import sorted_by_parenthood
from config.configuration_manager import current_configuration_manager, p4studio_arg_to_config_option
from config.configure_command import _allowed_options, configure_command
from dependencies.dependencies_command import install_command
from system.check_system_command import check_system_command
from utils.default_directory_file import DefaultDirectoryFile
from utils.log import logging_options, default_log_file_name
from utils.terminal import print_green, print_separator
from workspace import current_workspace
from .profile import Profile, load_profile_from_file
from .profile_execution_plan import ProfileExecutionPlan


@click.group("profile", short_help="Build and install SDE using profiles")
def profile_command() -> None:
    """Build and install SDE using profiles

    \b
    If you want to build and install SDE using profile profiles/switch-p4-16.yaml, run:
      p4studio profile apply profiles/switch-p4-16.yaml
    or:
      p4studio profile apply switch-p4-16

    """


def _default_options() -> str:
    options = {
        o.p4studio_name: o.default
        for o in current_configuration_manager().definitions
    }
    options['switch'] = True
    options['bf-diags'] = True

    return ','.join([('' if v else '^') + o for o, v in options.items()])


@click.command('create')
@click.argument("file", type=click.File('w'))
@click.option('--configure', 'options', default=None,
              help="Configure profile with comma separated list of options"
              )
@click.option('--switch-profile', help="Switch profile")
@click.option('--p4-examples', default="tna_exact_match", help="Comma separated list of P4 programs")
@click.option("--bsp-path", type=click.Path(exists=True), help="BSP to be used and installed")
def profile_create_command(file: LazyFile,
                           options: str,
                           switch_profile: str,
                           p4_examples: str,
                           bsp_path: Optional[str]
                           ) -> None:
    """
    Create new profile
    """
    config_manager = current_configuration_manager()
    profile = Profile(config_manager)

    if bsp_path:
        profile.bsp_path = str(bsp_path)

    if options is None:
        options = _default_options()

    config_options = config_manager.convert_to_config_options(options.split(','))
    for opt in sorted_by_parenthood(config_options):
        profile.set_option(opt.p4studio_name, opt.enabled)

    if switch_profile:
        profile.switch_profile = switch_profile

    if p4_examples:
        for program in p4_examples.split(','):
            profile.add_p4_program(program)

    file.write(yaml.dump(profile.raw))


@click.command('describe')
@click.argument("file",
                type=DefaultDirectoryFile(default_directory=str(current_workspace().p4studio_profiles_dir),
                                          accepted_extensions=[".yaml", ".yml"], logging_name="profile", mode='r'))
@click.option("--bsp-path", type=click.Path(exists=True), help="BSP to be used and installed")
def profile_describe_command(file: LazyFile, bsp_path: Optional[str]) -> None:
    """Describe existing profile"""
    plan = create_plan(file, bsp_path)

    plan.describe_profile()
    plan.show_commands()


def profile_file_autocompletion(ctx: Context, args: List[str], incomplete: str) -> List[str]:
    files_in_profile_dir = current_workspace().p4studio_profiles_dir.iterdir()
    yaml_file_names = [f.stem for f in files_in_profile_dir if f.suffix == ".yaml" or f.suffix == ".yml"]
    return [fn for fn in yaml_file_names if incomplete in fn]


@click.command('apply')
@click.argument("file",
                type=DefaultDirectoryFile(default_directory=str(current_workspace().p4studio_profiles_dir),
                                          accepted_extensions=[".yaml", ".yml"], logging_name="profile", mode='r',
                                          ), autocompletion=profile_file_autocompletion)
@logging_options('INFO', default_log_file_name())
@click.option("--override-option", "override_options", type=Choice(_allowed_options()), multiple=True,
              metavar="[CONFIG|^CONFIG]", help="Override any option in a profile")
@click.option("--jobs", default=os.cpu_count(), help="Allow specific number of jobs to be used")
@click.option("--bsp-path", type=click.Path(exists=True), help="BSP to be used and installed")
@click.option('--skip-dependencies', default=False, is_flag=True, help="Do not install dependencies")
@click.option('--skip-system-check', default=False, is_flag=True, help="Do not check system")
@click.pass_context
def profile_apply_command(context: Context,
                          file: LazyFile,
                          jobs: Optional[int],
                          bsp_path: Optional[str],
                          skip_dependencies: bool,
                          skip_system_check: bool,
                          override_options: List[str]
                          ) -> None:
    """
    Build and install SDE using existing profile
    """
    plan = create_plan(file, bsp_path, jobs, override_options)

    if not skip_system_check:
        check_system(context, plan)

    execute_plan(context, plan, skip_dependencies)


def check_system(context: Context, plan: ProfileExecutionPlan) -> None:
    asic = 'asic' in plan.profile.config_args()
    kdir = plan.profile.kdir
    context.invoke(check_system_command, asic=asic, kdir=kdir)


def execute_plan(context: Context, plan: ProfileExecutionPlan, skip_dependencies: bool = False) -> None:
    print_separator()
    plan.describe_profile()
    print_separator()

    if not skip_dependencies:
        context.invoke(install_command, **plan.dependencies_install_args())
        print_separator()

    context.invoke(configure_command, **plan.configure_args())
    print_separator()
    context.invoke(build_command, **plan.build_args())


def create_plan(file: LazyFile, bsp_path: Optional[str] = None, jobs: Optional[int] = None,
                overriden_options: Sequence[str] = []) -> ProfileExecutionPlan:
    print_green("Loading profile from {} file...", file.name)
    profile = load_profile_from_file(file)

    for option in sorted_by_parenthood([p4studio_arg_to_config_option(o) for o in overriden_options]):
        if option.enabled:
            profile.enable(option.p4studio_name)
        elif option.can_be_disabled():
            profile.disable(option.p4studio_name)

    if bsp_path:
        profile.enable("bsp")

    plan = ProfileExecutionPlan(profile, bsp_path, jobs)
    print_green("Profile is correct.")
    print_separator()
    return plan


profile_command.add_command(profile_create_command)
profile_command.add_command(profile_apply_command)
profile_command.add_command(profile_describe_command)

yaml.add_representer(OrderedDict, lambda self, data: SafeRepresenter.represent_dict(self, data.items()))
