#!/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

from build import build_command
from config import configure_command
from dependencies.dependencies_command import install_command
from profile.profile import Profile
from utils.click import command_call_to_str
from utils.terminal import print_green, print_normal, print_separator
from typing import Optional, Dict, Union, cast


class ProfileExecutionPlan:
    def __init__(self, profile: Profile, bsp_path: Optional[str], jobs: Optional[int]):
        self.profile = profile
        self.bsp_path = bsp_path or profile.bsp_path
        self.jobs = jobs

    def describe_profile(self) -> None:
        print_green('Source packages to install: ')
        for package in self.profile.source_packages():
            print_normal(" - {}", package)

        print_green('Configuration options: ')
        for option, enabled in self.profile.config_options().items():
            print_normal(" {} {}", '✓' if enabled else '✗', option)

        print_green('Targets to build: ')
        for target in self.profile.build_targets():
            print_normal(" - {}", target)

        print_separator()

    def show_commands(self) -> None:
        commands = [
            command_call_to_str(install_command, **self.dependencies_install_args()),
            command_call_to_str(configure_command, **self.configure_args()),
            command_call_to_str(build_command, **self.build_args()),
        ]

        print_green("Profile is equivalent to below list of commands:")
        for command in commands:
            print_normal(command)

    def dependencies_install_args(self) -> Dict[str, Union[str, int]]:
        result = cast(Dict[str, Union[str, int]], {
            'source_packages': ','.join(self.profile.source_packages()),
        })

        if self.jobs:
            result['jobs'] = self.jobs
        return result

    def configure_args(self) -> Dict[str, object]:
        return {
            'options': tuple(self.profile.config_args()),
            'bsp_path': self.bsp_path,
            'p4ppflags': self.profile.p4ppflags,
            'p4flags': self.profile.p4flags,
            'extra_cppflags': self.profile.extra_cppflags,
            'kdir': self.profile.kdir
        }

    def build_args(self) -> Dict[str, object]:
        result = cast(Dict[str, object], {
            'targets': tuple(self.profile.build_targets()),
        })
        if self.jobs:
            result['jobs'] = self.jobs
        return result
