#!/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

from pathlib import Path
from typing import Optional

import click

from system.check_system_utils import print_multiple_checks
from system.checks import check_kernel_headers, get_initial_checks
from utils.exceptions import ApplicationException
from utils.terminal import print_green


@click.command("check-system", short_help="Verify that system is capable to build and install SDE")
@click.option("--install-dir", help="Directory where SDE should be installed")
@click.option("--asic", is_flag=True, default=False, help="Check if system can be used to build for ASIC")
@click.option('--kdir', type=click.Path(file_okay=False), help="Path to Kernel headers")
def check_system_command(install_dir: Optional[str], asic: bool, kdir: Optional[str]) -> None:
    """
    Perform basic checks to verify that system is capable to build and install SDE
    """

    checks = get_initial_checks(Path(install_dir) if install_dir is not None else None)

    if asic:
        checks.append(check_kernel_headers(Path(kdir) if kdir else None))

    print_multiple_checks(checks)

    if any(not check.ok for check in checks):
        raise ApplicationException("At least one check failed")
    else:
        print_green("Checking system completed successfully.")
