#!/usr/bin/env python3


#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2022 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

from os import statvfs
from pathlib import Path
from shutil import which
from typing import List, Optional

from dependencies.dependency_manager import is_os_supported
from system.check_system_utils import CheckResult
from utils.os_info import get_system_kdir_path, os_info
from workspace import current_workspace

BASIC_TOOLS = ["sudo"]


def get_initial_checks(install_dir: Optional[Path] = None) -> List[CheckResult]:
    if not install_dir:
        install_dir = current_workspace().p4studio_path
    checks = [check_disk_space(install_dir), check_tmp_space(), check_os(), check_basic_tools_are_installed()]
    return checks


def check_disk_space(path: Path) -> CheckResult:
    fs = statvfs(str(path))
    free_space_gb = fs.f_bsize * fs.f_bavail / 1024 / 1024 / 1024
    min_free_space_gb = 20

    name = "Free space >= {}GB".format(min_free_space_gb)
    info = "{:.2f}GB".format(free_space_gb)
    ok = free_space_gb >= min_free_space_gb
    return CheckResult(name, info, ok)


def check_tmp_space() -> CheckResult:
    fs = statvfs('/tmp')
    free_space = fs.f_bsize * fs.f_bavail / 1024 / 1024 / 1024
    min_tmp_space = 2.5
    name = "Free space in /tmp >= {}GB".format(min_tmp_space)
    info = "{:.2f}GB".format(free_space)
    ok = free_space >= min_tmp_space
    return CheckResult(name, info, ok)


def check_os() -> CheckResult:
    name = "OS is supported"
    ok = is_os_supported(os_info.name, os_info.version)
    info = "{} {}".format(os_info.name, os_info.version)

    return CheckResult(name, info, ok)


def check_kernel_headers(kdir: Optional[Path]) -> CheckResult:
    if not kdir:
        kdir = get_system_kdir_path()
    ok = kdir.exists()
    info = str(kdir) + (" exists" if ok else " not exist")

    return CheckResult("Kernel headers installed", info, ok)


def check_basic_tools_are_installed() -> CheckResult:
    name = "Basic tools are installed"
    basic_tools_existence = [(basic_tool, bool(which(basic_tool))) for basic_tool in BASIC_TOOLS]
    ok = all([do_exist for _, do_exist in basic_tools_existence])
    info = ", ".join(["{} {}".format(tool, ('✓' if do_exist else '✗')) for tool, do_exist in basic_tools_existence])

    return CheckResult(name, info, ok)
