#!/usr/bin/env python3


#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.

from collections import OrderedDict
from itertools import groupby
from typing import Dict, Any, Callable, List, TypeVar, Sequence, Tuple

T = TypeVar('T')


def group_by_to_dict(items: List[T], key_function: Callable[[T], object]) -> OrderedDict:
    result = OrderedDict()
    for k, value in groupby(items, key_function):
        if k not in result:
            result[k] = []
        result[k] += list(value)
    return result


def nested_get(dictionary: Dict[str, Any], path: str, default: object) -> object:
    segments = path.split('/')
    current = dictionary
    current_path = []
    for segment in segments:
        current_path.append(segment)
        if not isinstance(current, dict):
            raise _not_dict_exception(current_path)
        if segment not in current:
            return default
        current = current[segment]
    return current


def nested_set(dictionary: Dict[str, Any], path: str, value: object) -> Dict[str, Any]:
    segments = path.split('/')
    current = dictionary
    current_path = []
    for segment in segments[:-1]:
        current_path.append(segment)
        if not isinstance(current, dict):
            raise _not_dict_exception(current_path)
        current = current.setdefault(segment, OrderedDict())

    if not isinstance(current, dict):
        raise _not_dict_exception(current_path)
    current[segments[-1]] = value
    return dictionary


def nested_del(dictionary: Dict[str, Any], path: str) -> Dict[str, Any]:
    segments = path.split('/')
    current = dictionary
    current_path = []
    for segment in segments[:-1]:
        if not isinstance(current, dict):
            raise _not_dict_exception(current_path)
        if segment not in current:
            return dictionary
        current_path.append(segment)
        current = current[segment]

    if not isinstance(current, dict):
        raise _not_dict_exception(current_path)
    del current[segments[-1]]
    return dictionary


def _not_dict_exception(path: Sequence[str]) -> Exception:
    message = "value of '{}' should be an instance of dict".format('/'.join(path))
    return Exception(message)


def partition(collection: List[T], predicate: Callable[[T], bool]) -> Tuple[List[T], List[T]]:
    a = []
    b = []
    for item in collection:
        if predicate(item):
            a.append(item)
        else:
            b.append(item)
    return a, b


def flatten(nested_list: List[List[T]]) -> List[T]:
    return [element for inside_list in nested_list for element in inside_list]


def diff(list1: List[T], list2: List[T]) -> List[T]:
    return [x for x in list1 if x not in list2]


def as_list_of_strings(objects: Sequence[object]) -> List[str]:
    return [str(x) for x in objects]
