# !/usr/bin/env python3

#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.
import os
import subprocess
from datetime import timedelta
from pathlib import Path
from shlex import split
from timeit import default_timer as timer
from typing import List, Sequence, Union, Mapping, Dict, cast, IO, AnyStr, Optional

from utils.exceptions import ApplicationException
from utils.types import PathLike

try:
    from shlex import quote as cmd_quote
except ImportError:
    from pipes import quote as cmd_quote


def execute(command: Union[str, Sequence[str]], working_dir: Optional[PathLike] = None,
            override_env: Dict[str, str] = None) -> None:
    if not try_execute(command, working_dir=working_dir, override_env=override_env):
        raise ApplicationException("Failed to run: {}".format(command))


def try_execute(command: Union[str, Sequence[str]], working_dir: Optional[PathLike] = None,
                override_env: Dict[str, str] = None) -> bool:
    from utils.terminal import compact_log

    if isinstance(command, str):
        command = split(command)

    env = os.environ.copy()
    if override_env:
        env.update(override_env)

    try:
        if working_dir is None:
            working_dir = Path.cwd()
            compact_log().log("Executing: {}".format(' '.join(command)))
        else:
            compact_log().log("Executing: '{}' in {}".format(' '.join(command), working_dir))
        start = timer()
        process = run_subprocess(command, working_dir, env)

        while process.poll() is None:
            while True:
                stdout = process.stdout
                line = stdout.readline().decode().rstrip() if stdout else ""
                if line:
                    compact_log().log(line)
                else:
                    break

        end = timer()
        compact_log().log("Cmd '{}' took: {}, status: {}".format(' '.join(command), timedelta(seconds=end - start),
                                                                 process.returncode))
        success = process.returncode == 0

        if success:
            compact_log().reset(include_last=False)
        else:
            compact_log().dump()

        return success

    except FileNotFoundError as e:
        raise ApplicationException from e


def run_subprocess(command: Union[str, Sequence[str]], working_dir: PathLike = Path.cwd(),
                   env: Mapping[str, str] = None) -> subprocess.Popen:
    return subprocess.Popen(
        split(command) if isinstance(command, str) else command,
        cwd=str(working_dir),
        stdout=subprocess.PIPE,
        stderr=subprocess.STDOUT,
        env=env
    )


def command_output(command: List[str]) -> bytes:
    try:
        return subprocess.check_output(command)
    except subprocess.CalledProcessError as e:
        raise ApplicationException from e


def check_command(command: List[str], override_env: Mapping[str, str] = None) -> bool:
    env = os.environ.copy()
    if override_env:
        env.update(override_env)
    process = run_subprocess(command, env=env)
    process.communicate()
    return process.returncode == 0


def cmd_args_to_str(args: Sequence[str]) -> str:
    return " ".join(map(cmd_quote, args))
