#!/usr/bin/env python3


#  INTEL CONFIDENTIAL
#
#  Copyright (c) 2021 Intel Corporation
#  All Rights Reserved.
#
#  This software and the related documents are Intel copyrighted materials,
#  and your use of them is governed by the express license under which they
#  were provided to you ("License"). Unless the License provides otherwise,
#  you may not use, modify, copy, publish, distribute, disclose or transmit this
#  software or the related documents without Intel's prior written permission.
#
#  This software and the related documents are provided as is, with no express or
#  implied warranties, other than those that are expressly stated in the License.
from pathlib import Path
from typing import List

from utils.exceptions import ApplicationException
from utils.processes import try_execute
from utils.terminal import compact_log


class SubprocessBuilder:
    def __init__(self, command_name: str = None):
        self._name = command_name
        self._args = []

    def args(self, *args: object) -> 'SubprocessBuilder':
        self._args += [str(arg) for arg in args]
        return self

    def sudo(self, home: bool = False) -> 'SubprocessBuilder':
        return self.args('sudo', '-HE') if home else self.args('sudo', '-E')

    def pip3(self) -> 'SubprocessBuilder':
        return self.args('env', 'pip3')

    def pip3_install(self, packages: List[str]) -> 'SubprocessBuilder':
        packages = packages or []
        return self.pip3().args('install').args(*packages)

    def python3(self, script: Path) -> 'SubprocessBuilder':
        return self.args('env', 'python3', script)

    def run_or_raise(self, failure_message: str = None, working_dir: Path = Path.cwd()) -> None:
        name = self._name or "running {}".format(self._args[0])
        compact_log().start_new("- {}: ".format(name))
        if not try_execute(self._args, working_dir):
            if not failure_message:
                failure_message = "Problem occurred while {}".format(name)
            compact_log().log("\n{}\n".format(failure_message))
            compact_log().done(False)
            raise ApplicationException(failure_message.format(' '.join(self._args)))
        compact_log().done(True)


def subprocess_builder(command_name: str = None) -> SubprocessBuilder:
    return SubprocessBuilder(command_name)
