/*******************************************************************************
 * BAREFOOT NETWORKS CONFIDENTIAL & PROPRIETARY
 *
 * Copyright (c) 2019-present Barefoot Networks, Inc.
 *
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Barefoot Networks, Inc. and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Barefoot Networks, Inc.
 * and its suppliers and may be covered by U.S. and Foreign Patents, patents in
 * process, and are protected by trade secret or copyright law.  Dissemination of
 * this information or reproduction of this material is strictly forbidden unless
 * prior written permission is obtained from Barefoot Networks, Inc.
 *
 * No warranty, explicit or implicit is provided, unless granted under a written
 * agreement with Barefoot Networks, Inc.
 *
 ******************************************************************************/

#include <core.p4>
#if __TARGET_TOFINO__ == 2
#include <t2na.p4>
#define CPU_PORT_VALUE 2
#else
#include <tna.p4>
#define CPU_PORT_VALUE 64
#endif

#include "common/headers_amlight.p4"
#include "common/util.p4"

const PortId_t CPU_PORT = CPU_PORT_VALUE;

// Constants for header sizes (from headers.h)
const bit<32> TELEMETRY_REPORT_SIZE = 16; // From headers.h
const bit<32> INT_SHIM_SIZE = 4;         // From headers.h
const bit<32> INT_METADATA_SIZE = 8;     // From headers.h
const bit<32> INT_STACK_SIZE = 24;       // From headers.h

/* Digest structure aligned with parsed_packet_data_t in packet_parser.c */
struct digest_a_t {
    ipv4_addr_t src_addr;        // Inner source IP
    ipv4_addr_t dst_addr;        // Inner destination IP
    bit<8> protocol;             // Inner protocol
    bit<16> src_port;            // Inner source port
    bit<16> dst_port;            // Inner destination port
    bit<16> packet_length;       // Inner IPv4 total length
    bit<24> queue_occupancy;     // From int_stack
    bit<32> ingress_timestamp;   // From int_stack
    bit<32> egress_timestamp;    // From int_stack
}

struct metadata_t {
    bit<17> flowID;              // Flow ID for register indexing
    bit<16> src_port;            // Inner source port
    bit<16> dst_port;            // Inner destination port
    bit<8> protocol;             // Inner protocol
    bit<32> ingress_timestamp;   // From int_stack
    bit<32> egress_timestamp;    // From int_stack
    bit<24> queue_occupancy;     // From int_stack
    bit<2> outer_vlan_count;     // Number of outer VLAN tags
    bit<2> inner_vlan_count;     // Number of inner VLAN tags
}

// ---------------------------------------------------------------------------
// Ingress Parser
// ---------------------------------------------------------------------------
parser SwitchIngressParser(
        packet_in pkt,
        out header_t hdr,
        out metadata_t ig_md,
        out ingress_intrinsic_metadata_t ig_intr_md) {

    TofinoIngressParser() tofino_parser;

    state start {
        tofino_parser.apply(pkt, ig_intr_md);
        ig_md.src_port = 0;
        ig_md.dst_port = 0;
        ig_md.protocol = 0;
        ig_md.flowID = 0;
        ig_md.ingress_timestamp = 0;
        ig_md.egress_timestamp = 0;
        ig_md.queue_occupancy = 0;
        ig_md.outer_vlan_count = 0;
        ig_md.inner_vlan_count = 0;
        transition parse_outer_ethernet;
    }

    state parse_outer_ethernet {
        pkt.extract(hdr.outer_ethernet);
        transition select (hdr.outer_ethernet.ether_type) {
            ETHERTYPE_VLAN: parse_outer_vlan1;
            ETHERTYPE_QINQ: parse_outer_vlan1;
            ETHERTYPE_IPV4: parse_outer_ipv4;
            default: reject;
        }
    }

    state parse_outer_vlan1 {
        pkt.extract(hdr.outer_vlan1);
        ig_md.outer_vlan_count = 1;
        transition select (hdr.outer_vlan1.ether_type) {
            ETHERTYPE_VLAN: parse_outer_vlan2;
            ETHERTYPE_QINQ: parse_outer_vlan2;
            ETHERTYPE_IPV4: parse_outer_ipv4;
            default: reject;
        }
    }

    state parse_outer_vlan2 {
        pkt.extract(hdr.outer_vlan2);
        ig_md.outer_vlan_count = 2;
        transition parse_outer_ipv4;
    }

    state parse_outer_ipv4 {
        pkt.extract(hdr.outer_ipv4);
        transition select (hdr.outer_ipv4.protocol) {
            IP_PROTOCOLS_TCP: parse_outer_tcp;
            IP_PROTOCOLS_UDP: parse_outer_udp;
            default: reject;
        }
    }

    state parse_outer_tcp {
        pkt.extract(hdr.outer_tcp);
        transition parse_telemetry_report;
    }

    state parse_outer_udp {
        pkt.extract(hdr.outer_udp);
        transition parse_telemetry_report;
    }

    state parse_telemetry_report {
        pkt.extract(hdr.telemetry_report);
        transition parse_inner_ethernet;
    }

    state parse_inner_ethernet {
        pkt.extract(hdr.inner_ethernet);
        transition select (hdr.inner_ethernet.ether_type) {
            ETHERTYPE_VLAN: parse_inner_vlan1;
            ETHERTYPE_QINQ: parse_inner_vlan1;
            ETHERTYPE_IPV4: parse_inner_ipv4;
            default: reject;
        }
    }

    state parse_inner_vlan1 {
        pkt.extract(hdr.inner_vlan1);
        ig_md.inner_vlan_count = 1;
        transition select (hdr.inner_vlan1.ether_type) {
            ETHERTYPE_VLAN: parse_inner_vlan2;
            ETHERTYPE_QINQ: parse_inner_vlan2;
            ETHERTYPE_IPV4: parse_inner_ipv4;
            default: reject;
        }
    }

    state parse_inner_vlan2 {
        pkt.extract(hdr.inner_vlan2);
        ig_md.inner_vlan_count = 2;
        transition parse_inner_ipv4;
    }

    state parse_inner_ipv4 {
        pkt.extract(hdr.inner_ipv4);
        ig_md.protocol = hdr.inner_ipv4.protocol;
        transition select (hdr.inner_ipv4.protocol) {
            IP_PROTOCOLS_TCP: parse_inner_tcp;
            IP_PROTOCOLS_UDP: parse_inner_udp;
            default: accept;
        }
    }

    state parse_inner_tcp {
        pkt.extract(hdr.inner_tcp);
        ig_md.src_port = hdr.inner_tcp.src_port;
        ig_md.dst_port = hdr.inner_tcp.dst_port;
        transition parse_int_shim;
    }

    state parse_inner_udp {
        pkt.extract(hdr.inner_udp);
        ig_md.src_port = hdr.inner_udp.src_port;
        ig_md.dst_port = hdr.inner_udp.dst_port;
        transition parse_int_shim;
    }

    state parse_int_shim {
        pkt.extract(hdr.int_shim);
        transition parse_int_metadata;
    }

    state parse_int_metadata {
        pkt.extract(hdr.int_metadata);
        transition parse_int_stack;
    }

    state parse_int_stack {
        pkt.extract(hdr.int_stack);
        ig_md.ingress_timestamp = hdr.int_stack.ingress_timestamp;
        ig_md.egress_timestamp = hdr.int_stack.egress_timestamp;
        ig_md.queue_occupancy = hdr.int_stack.queue_occupancy;
        transition accept;
    }
}

// ---------------------------------------------------------------------------
// Ingress Deparser
// ---------------------------------------------------------------------------
control SwitchIngressDeparser(
        packet_out pkt,
        inout header_t hdr,
        in metadata_t ig_md,
        in ingress_intrinsic_metadata_for_deparser_t ig_dprsr_md) {
    Digest<digest_a_t>() digest_a;

    apply {
        // Generate a digest for packets with digest_type set
        if (ig_dprsr_md.digest_type == 1) {
            digest_a.pack({
                hdr.inner_ipv4.src_addr,
                hdr.inner_ipv4.dst_addr,
                ig_md.protocol,
                ig_md.src_port,
                ig_md.dst_port,
                hdr.inner_ipv4.total_len,
                ig_md.queue_occupancy,
                ig_md.ingress_timestamp,
                ig_md.egress_timestamp
            });
        }

        pkt.emit(hdr);
    }
}

// ---------------------------------------------------------------------------
// Ingress Control
// ---------------------------------------------------------------------------
control SwitchIngress(
        inout header_t hdr,
        inout metadata_t md,
        in ingress_intrinsic_metadata_t ig_intr_md,
        in ingress_intrinsic_metadata_from_parser_t ig_prsr_md,
        inout ingress_intrinsic_metadata_for_deparser_t ig_dprsr_md,
        inout ingress_intrinsic_metadata_for_tm_t ig_tm_md) {

    // Hash for flow ID (includes src_addr, dst_addr, and protocol as requested)
    Hash<bit<17>>(HashAlgorithm_t.CRC16) flow_hash;

    // Registers for flow tracking
    Register<bit<32>, bit<17>>(size=131072, initial_value=0) r_srcAddr;
    Register<bit<32>, bit<17>>(size=131072, initial_value=0) r_dstAddr;
    Register<bit<32>, bit<17>>(size=131072, initial_value=0) r_counter;

    RegisterAction<bit<32>, bit<17>, bit<32>>(r_srcAddr) r_srcAddr_update = {
        void apply(inout bit<32> value) {
            value = hdr.inner_ipv4.src_addr;
        }
    };

    RegisterAction<bit<32>, bit<17>, bit<32>>(r_dstAddr) r_dstAddr_update = {
        void apply(inout bit<32> value) {
            value = hdr.inner_ipv4.dst_addr;
        }
    };

    RegisterAction<bit<32>, bit<17>, bit<32>>(r_counter) r_counter_update = {
        void apply(inout bit<32> value) {
            value = value + 1; // Increment the counter
        }
    };

    // Digest control table
    action enable_digest() {
        ig_dprsr_md.digest_type = 1;
    }

    table control_digest {
        key = {
            hdr.inner_ipv4.dst_addr : exact;
        }
        actions = {
            enable_digest;
            NoAction;
        }
        const default_action = NoAction;
        size = 1024;
    }

    // Forwarding table
    action send_using_port(PortId_t egress_port) {
        ig_tm_md.ucast_egress_port = egress_port;
    }

    action drop() {
        ig_dprsr_md.drop_ctl = 0x1;
    }

    table forwarding {
        key = {
            ig_intr_md.ingress_port : exact;
        }
        actions = {
            send_using_port;
            drop;
        }
        const default_action = drop;
        size = 10;
    }

    apply {
        // Compute flow ID for register updates
        if (hdr.inner_ipv4.isValid()) {
            // Set protocol and port fields in metadata
            md.protocol = hdr.inner_ipv4.protocol;
            if (hdr.inner_ipv4.protocol == IP_PROTOCOLS_TCP && hdr.inner_tcp.isValid()) {
                md.src_port = hdr.inner_tcp.src_port;
                md.dst_port = hdr.inner_tcp.dst_port;
            } else if (hdr.inner_ipv4.protocol == IP_PROTOCOLS_UDP && hdr.inner_udp.isValid()) {
                md.src_port = hdr.inner_udp.src_port;
                md.dst_port = hdr.inner_udp.dst_port;
            } else {
                md.src_port = 0;
                md.dst_port = 0;
            }

            md.flowID = flow_hash.get({
                hdr.inner_ipv4.src_addr,
                hdr.inner_ipv4.dst_addr,
                hdr.inner_ipv4.protocol
            });

            // Update registers if flowID is non-zero
            if (md.flowID != 0) {
                r_srcAddr_update.execute(md.flowID);
                r_dstAddr_update.execute(md.flowID);
                r_counter_update.execute(md.flowID);
            }

            // Apply digest control table
            control_digest.apply();
        }

        // Apply forwarding table
        forwarding.apply();

        // Bypass egress processing
        ig_tm_md.bypass_egress = 1w1;
    }
}

Pipeline(SwitchIngressParser(),
         SwitchIngress(),
         SwitchIngressDeparser(),
         EmptyEgressParser(),
         EmptyEgress(),
         EmptyEgressDeparser()) pipe;

Switch(pipe) main;